<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Mara7el_Watermark_Cache_Manager
 * Handles caching of watermarked files.
 */
class Mara7el_Watermark_Cache_Manager {

	/**
	 * Cache directory path.
	 *
	 * @var string
	 */
	private $cache_dir;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$upload_dir = wp_upload_dir();
		$this->cache_dir = $upload_dir['basedir'] . '/mara7el-watermark-cache';
		
		if ( ! file_exists( $this->cache_dir ) ) {
			wp_mkdir_p( $this->cache_dir );
			// Add index.php and .htaccess to prevent listing
			file_put_contents( $this->cache_dir . '/index.php', '<?php // Silence is golden' );
			file_put_contents( $this->cache_dir . '/.htaccess', 'deny from all' );
		}
	}

	/**
	 * Get path to cached file if it exists and is valid.
	 *
	 * @param string $file_hash Unique hash for the file + user + settings combination.
	 * @return string|false Absolute path to file or false.
	 */
	public function get_cached_file( $file_hash, $extension = 'pdf' ) {
		$file_path = $this->cache_dir . '/' . $file_hash . '.' . $extension;

		if ( file_exists( $file_path ) ) {
			// Check expiration (e.g., 24 hours)
			// This could be configurable via settings
			$ttl = apply_filters( 'mara7el_watermark_cache_ttl', 24 * HOUR_IN_SECONDS );
			
			if ( ( time() - filemtime( $file_path ) ) < $ttl ) {
				return $file_path;
			} else {
				// Expired, delete it
				unlink( $file_path );
			}
		}

		return false;
	}

	/**
	 * Store a file in the cache.
	 *
	 * @param string $file_hash Unique hash.
	 * @param string $content   File content (binary).
	 * @param string $extension File extension.
	 * @return string|false Path to stored file.
	 */
	public function store_cached_file( $file_hash, $content, $extension = 'pdf' ) {
		$file_path = $this->cache_dir . '/' . $file_hash . '.' . $extension;
		
		if ( file_put_contents( $file_path, $content ) ) {
			return $file_path;
		}

		return false;
	}

	/**
	 * Clear all cached files.
	 */
	public function clear_cache() {
		$files = glob( $this->cache_dir . '/*' );
		foreach ( $files as $file ) {
			if ( is_file( $file ) && basename( $file ) !== 'index.php' && basename( $file ) !== '.htaccess' ) {
				unlink( $file );
			}
		}
	}
	
	/**
	 * Generate a cache key based on parameters.
	 * 
	 * @param string $file_path Original file path.
	 * @param int    $user_id   User ID.
	 * @param array  $settings  Watermark settings.
	 * @return string MD5 hash.
	 */
	public function generate_cache_key( $file_path, $user_id, $settings ) {
		return md5( $file_path . $user_id . serialize( $settings ) . MARA7EL_WATERMARK_VERSION );
	}
}
