<?php
/**
 * LMS Hooks Class.
 *
 * @package Mara7el_Live
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Mara7el_Live_LMS_Hooks {

	public function __construct() {
		add_action( 'add_meta_boxes', array( $this, 'add_live_session_meta_box' ) );
		add_action( 'save_post', array( $this, 'save_live_session_meta' ) );
		add_action( 'admin_notices', array( $this, 'check_lms_active' ) );
		
		// Frontend display
		add_filter( 'the_content', array( $this, 'display_live_session_info' ) );
	}

	/**
	 * Display Live Session Info on Frontend.
	 *
	 * @param string $content Post content.
	 * @return string Modified content.
	 */
	public function display_live_session_info( $content ) {
		// Only display on LearnPress courses and lessons
		if ( ! is_singular( array( 'lp_course', 'lp_lesson' ) ) ) {
			return $content;
		}

		$post_id  = get_the_ID();
		$join_url = get_post_meta( $post_id, '_mara7el_live_join_url', true );
		$start_time = get_post_meta( $post_id, '_mara7el_live_start_time', true );

		if ( ! $join_url ) {
			return $content;
		}

		// Check if user is enrolled (for courses) or has access (for lessons)
		$user_id = get_current_user_id();
		$has_access = false;

		if ( get_post_type() === 'lp_course' ) {
			// Check if user is enrolled in the course
			if ( function_exists( 'learn_press_user_enrolled_course' ) ) {
				$has_access = learn_press_user_enrolled_course( $post_id, $user_id );
			}
		} elseif ( get_post_type() === 'lp_lesson' ) {
			// Get parent course and check enrollment
			$course_id = get_post_meta( $post_id, '_lp_course', true );
			if ( $course_id && function_exists( 'learn_press_user_enrolled_course' ) ) {
				$has_access = learn_press_user_enrolled_course( $course_id, $user_id );
			}
		}

		// Allow administrators and instructors to always see the link
		if ( ! $has_access && ! current_user_can( 'manage_options' ) && ! current_user_can( 'lp_teacher' ) ) {
			return $content;
		}

		$html = '<div class="mara7el-live-session-box lp-box" style="background: #f9f9f9; padding: 20px; border: 1px solid #ddd; margin: 20px 0; border-radius: 5px;">';
		$html .= '<h3>' . __( 'Live Class Session', 'mara7el-live' ) . '</h3>';
		
		if ( $start_time ) {
			$formatted_time = date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $start_time ) );
			$html .= '<p><strong>' . __( 'Start Time:', 'mara7el-live' ) . '</strong> ' . $formatted_time . '</p>';
		}

		$html .= '<a href="' . esc_url( $join_url ) . '" class="button button-primary lp-button mara7el-join-btn" target="_blank" style="background-color: #0073aa; color: #fff; padding: 10px 20px; text-decoration: none; border-radius: 3px; display: inline-block;">' . __( 'Join Live Class', 'mara7el-live' ) . '</a>';
		$html .= '</div>';

		return $html . $content;
	}

	/**
	 * Check if LearnPress is active.
	 */
	public function check_lms_active() {
		// Check if LearnPress is active
		if ( ! function_exists( 'learn_press_get_version' ) ) {
			?>
			<div class="notice notice-error is-dismissible">
				<p><?php 
					printf(
						/* translators: %s: LearnPress plugin link */
						__( 'Mara7el Live Classes requires LearnPress to be installed and activated. Please install %s first.', 'mara7el-live' ),
						'<a href="https://wordpress.org/plugins/learnpress/" target="_blank">LearnPress</a>'
					);
				?></p>
			</div>
			<?php
		}
	}

	/**
	 * Add Meta Box.
	 */
	public function add_live_session_meta_box() {
		// Add meta box to LearnPress courses and lessons
		$screens = array( 'lp_course', 'lp_lesson' );
		
		foreach ( $screens as $screen ) {
			add_meta_box(
				'mara7el_live_session_box',
				__( 'Live Session Settings', 'mara7el-live' ),
				array( $this, 'render_meta_box' ),
				$screen,
				'normal',
				'high'
			);
		}
	}

	/**
	 * Render Meta Box.
	 *
	 * @param WP_Post $post Post object.
	 */
	public function render_meta_box( $post ) {
		wp_nonce_field( 'mara7el_live_save_session', 'mara7el_live_session_nonce' );

		$platform   = get_post_meta( $post->ID, '_mara7el_live_platform', true );
		$start_time = get_post_meta( $post->ID, '_mara7el_live_start_time', true );
		$duration   = get_post_meta( $post->ID, '_mara7el_live_duration', true );
		$join_url   = get_post_meta( $post->ID, '_mara7el_live_join_url', true );
		$start_url  = get_post_meta( $post->ID, '_mara7el_live_start_url', true );
		?>
		<p>
			<label for="mara7el_live_platform"><?php _e( 'Platform:', 'mara7el-live' ); ?></label>
			<select name="mara7el_live_platform" id="mara7el_live_platform" class="widefat">
				<option value=""><?php _e( 'Select Platform', 'mara7el-live' ); ?></option>
				<option value="zoom" <?php selected( $platform, 'zoom' ); ?>>Zoom</option>
				<option value="google_meet" <?php selected( $platform, 'google_meet' ); ?>>Google Meet</option>
				<option value="teams" <?php selected( $platform, 'teams' ); ?>>Microsoft Teams</option>
			</select>
		</p>
		<p>
			<label for="mara7el_live_start_time"><?php _e( 'Start Time:', 'mara7el-live' ); ?></label>
			<input type="datetime-local" name="mara7el_live_start_time" id="mara7el_live_start_time" value="<?php echo esc_attr( $start_time ); ?>" class="widefat" />
		</p>
		<p>
			<label for="mara7el_live_duration"><?php _e( 'Duration (minutes):', 'mara7el-live' ); ?></label>
			<input type="number" name="mara7el_live_duration" id="mara7el_live_duration" value="<?php echo esc_attr( $duration ); ?>" class="widefat" />
		</p>
		
		<?php if ( $join_url ) : ?>
			<div style="margin-top: 10px; padding: 10px; background: #f0f0f1; border: 1px solid #ccc;">
				<p><strong><?php _e( 'Join URL:', 'mara7el-live' ); ?></strong> <a href="<?php echo esc_url( $join_url ); ?>" target="_blank"><?php echo esc_html( $join_url ); ?></a></p>
				<?php if ( $start_url ) : ?>
					<p><strong><?php _e( 'Start URL (Host):', 'mara7el-live' ); ?></strong> <a href="<?php echo esc_url( $start_url ); ?>" target="_blank"><?php _e( 'Start Meeting', 'mara7el-live' ); ?></a></p>
				<?php endif; ?>
			</div>
		<?php else : ?>
			<p><em><?php _e( 'Save post to create the meeting link.', 'mara7el-live' ); ?></em></p>
		<?php endif; ?>
		<?php
	}

	/**
	 * Save Meta Box Data.
	 *
	 * @param int $post_id Post ID.
	 */
	public function save_live_session_meta( $post_id ) {
		if ( ! isset( $_POST['mara7el_live_session_nonce'] ) || ! wp_verify_nonce( $_POST['mara7el_live_session_nonce'], 'mara7el_live_save_session' ) ) {
			return;
		}

		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		// Save fields
		if ( isset( $_POST['mara7el_live_platform'] ) ) {
			update_post_meta( $post_id, '_mara7el_live_platform', sanitize_text_field( $_POST['mara7el_live_platform'] ) );
		}
		if ( isset( $_POST['mara7el_live_start_time'] ) ) {
			update_post_meta( $post_id, '_mara7el_live_start_time', sanitize_text_field( $_POST['mara7el_live_start_time'] ) );
		}
		if ( isset( $_POST['mara7el_live_duration'] ) ) {
			update_post_meta( $post_id, '_mara7el_live_duration', intval( $_POST['mara7el_live_duration'] ) );
		}

		// Trigger session creation if platform is selected and no URL exists yet
		$platform = get_post_meta( $post_id, '_mara7el_live_platform', true );
		$join_url = get_post_meta( $post_id, '_mara7el_live_join_url', true );

		if ( $platform && ! $join_url ) {
			$session_manager = new Mara7el_Live_Session_Manager();
			$session_manager->create_session( $post_id );
		}
	}
}
