<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Mara7el_Watermark_Logging
 * Handles database logging of access events.
 */
class Mara7el_Watermark_Logging {

	/**
	 * Table name.
	 *
	 * @var string
	 */
	private static $table_name;

	/**
	 * Constructor.
	 */
	public function __construct() {
		global $wpdb;
		self::$table_name = $wpdb->prefix . 'mara7el_access_logs';
	}

	/**
	 * Create the logging table.
	 */
	public static function create_tables() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'mara7el_access_logs';
		$charset_collate = $wpdb->get_charset_collate();

		$sql = "CREATE TABLE $table_name (
			id bigint(20) NOT NULL AUTO_INCREMENT,
			user_id bigint(20) NOT NULL,
			file_id varchar(255) NOT NULL,
			action varchar(50) NOT NULL,
			ip_address varchar(100) NOT NULL,
			user_agent text NOT NULL,
			session_id varchar(100) DEFAULT '',
			meta_data longtext DEFAULT '',
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY  (id),
			KEY user_id (user_id),
			KEY file_id (file_id),
			KEY action (action)
		) $charset_collate;";

		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
		dbDelta( $sql );
	}

	/**
	 * Log an event.
	 *
	 * @param int    $user_id User ID.
	 * @param string $file_id File identifier (path or ID).
	 * @param string $action  Action type (view, download, error).
	 * @param array  $meta    Additional metadata.
	 * @return int|false The inserted ID or false on error.
	 */
	public function log_access( $user_id, $file_id, $action, $meta = array() ) {
		global $wpdb;

		$data = array(
			'user_id'    => $user_id,
			'file_id'    => $file_id,
			'action'     => $action,
			'ip_address' => $this->get_client_ip(),
			'user_agent' => isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '',
			'session_id' => wp_get_session_token(),
			'meta_data'  => maybe_serialize( $meta ),
			'created_at' => current_time( 'mysql' ),
		);

		$format = array( '%d', '%s', '%s', '%s', '%s', '%s', '%s', '%s' );

		$result = $wpdb->insert( self::$table_name, $data, $format );

		if ( $result ) {
			return $wpdb->insert_id;
		}

		return false;
	}

	/**
	 * Get logs with filtering.
	 *
	 * @param array $args Filter arguments.
	 * @return array List of log entries.
	 */
	public function get_logs( $args = array() ) {
		global $wpdb;

		$defaults = array(
			'user_id' => '',
			'file_id' => '',
			'action'  => '',
			'limit'   => 50,
			'offset'  => 0,
			'orderby' => 'created_at',
			'order'   => 'DESC',
		);

		$args = wp_parse_args( $args, $defaults );
		$where = 'WHERE 1=1';
		$query_args = array();

		if ( ! empty( $args['user_id'] ) ) {
			$where .= ' AND user_id = %d';
			$query_args[] = $args['user_id'];
		}

		if ( ! empty( $args['file_id'] ) ) {
			$where .= ' AND file_id = %s';
			$query_args[] = $args['file_id'];
		}

		if ( ! empty( $args['action'] ) ) {
			$where .= ' AND action = %s';
			$query_args[] = $args['action'];
		}

		$limit  = absint( $args['limit'] );
		$offset = absint( $args['offset'] );
		$order  = strtoupper( $args['order'] ) === 'ASC' ? 'ASC' : 'DESC';
		$orderby = sanitize_sql_orderby( $args['orderby'] );

		$sql = "SELECT * FROM " . self::$table_name . " $where ORDER BY $orderby $order LIMIT %d OFFSET %d";
		$query_args[] = $limit;
		$query_args[] = $offset;

		return $wpdb->get_results( $wpdb->prepare( $sql, $query_args ) );
	}

	/**
	 * Get client IP address.
	 *
	 * @return string
	 */
	private function get_client_ip() {
		$ip = '';
		if ( isset( $_SERVER['HTTP_CLIENT_IP'] ) ) {
			$ip = $_SERVER['HTTP_CLIENT_IP'];
		} elseif ( isset( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
			$ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
		} elseif ( isset( $_SERVER['REMOTE_ADDR'] ) ) {
			$ip = $_SERVER['REMOTE_ADDR'];
		}
		return sanitize_text_field( wp_unslash( $ip ) );
	}
}
