<?php
/**
 * REST API Class.
 *
 * @package Mara7el_Live
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Mara7el_Live_REST_API {

	public function __construct() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
	}

	/**
	 * Register REST API routes.
	 */
	public function register_routes() {
		register_rest_route(
			'mara7el-live/v1',
			'/create',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'create_session' ),
				'permission_callback' => array( $this, 'check_permission' ),
			)
		);

		register_rest_route(
			'mara7el-live/v1',
			'/list',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'list_sessions' ),
				'permission_callback' => array( $this, 'check_permission' ),
			)
		);

		register_rest_route(
			'mara7el-live/v1',
			'/delete',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'delete_session' ),
				'permission_callback' => array( $this, 'check_permission' ),
			)
		);
	}

	/**
	 * Check permissions.
	 * Allow administrators and LearnPress instructors.
	 */
	public function check_permission() {
		return current_user_can( 'manage_options' ) || current_user_can( 'lp_teacher' );
	}

	/**
	 * Create Session via API.
	 *
	 * @param WP_REST_Request $request Request object.
	 */
	public function create_session( $request ) {
		$post_id = $request->get_param( 'post_id' );
		
		if ( ! $post_id ) {
			return new WP_Error( 'missing_param', 'post_id is required', array( 'status' => 400 ) );
		}

		// Update meta first
		$platform = $request->get_param( 'platform' );
		if ( $platform ) {
			update_post_meta( $post_id, '_mara7el_live_platform', sanitize_text_field( $platform ) );
		}
		
		$start_time = $request->get_param( 'start_time' );
		if ( $start_time ) {
			update_post_meta( $post_id, '_mara7el_live_start_time', sanitize_text_field( $start_time ) );
		}

		$duration = $request->get_param( 'duration' );
		if ( $duration ) {
			update_post_meta( $post_id, '_mara7el_live_duration', intval( $duration ) );
		}

		// Create session
		$session_manager = new Mara7el_Live_Session_Manager();
		$session_manager->create_session( $post_id );

		return rest_ensure_response( array(
			'success' => true,
			'message' => 'Session created',
			'data'    => array(
				'join_url'  => get_post_meta( $post_id, '_mara7el_live_join_url', true ),
				'start_url' => get_post_meta( $post_id, '_mara7el_live_start_url', true ),
			),
		) );
	}

	/**
	 * List Sessions.
	 */
	public function list_sessions() {
		$args = array(
			'post_type'  => array( 'lp_course', 'lp_lesson' ),
			'meta_query' => array(
				array(
					'key'     => '_mara7el_live_session_id',
					'compare' => 'EXISTS',
				),
			),
			'posts_per_page' => -1,
		);

		$query = new WP_Query( $args );
		$sessions = array();

		if ( $query->have_posts() ) {
			while ( $query->have_posts() ) {
				$query->the_post();
				$sessions[] = array(
					'post_id'    => get_the_ID(),
					'title'      => get_the_title(),
					'post_type'  => get_post_type(),
					'platform'   => get_post_meta( get_the_ID(), '_mara7el_live_platform', true ),
					'start_time' => get_post_meta( get_the_ID(), '_mara7el_live_start_time', true ),
					'join_url'   => get_post_meta( get_the_ID(), '_mara7el_live_join_url', true ),
				);
			}
			wp_reset_postdata();
		}

		return rest_ensure_response( $sessions );
	}

	/**
	 * Delete Session.
	 *
	 * @param WP_REST_Request $request Request object.
	 */
	public function delete_session( $request ) {
		$post_id = $request->get_param( 'post_id' );
		
		if ( ! $post_id ) {
			return new WP_Error( 'missing_param', 'post_id is required', array( 'status' => 400 ) );
		}

		delete_post_meta( $post_id, '_mara7el_live_session_id' );
		delete_post_meta( $post_id, '_mara7el_live_join_url' );
		delete_post_meta( $post_id, '_mara7el_live_start_url' );
		delete_post_meta( $post_id, '_mara7el_live_platform' );
		delete_post_meta( $post_id, '_mara7el_live_start_time' );

		return rest_ensure_response( array( 'success' => true, 'message' => 'Session deleted locally' ) );
	}
}
