<?php

class Mara7el_Booking_Schedule_Engine {

	/**
	 * Check if a teacher is available at a specific time.
	 *
	 * @param int $teacher_id
	 * @param string $datetime Y-m-d H:i:s
	 * @param int $duration_minutes
	 * @return bool
	 */
	public function is_teacher_available( $teacher_id, $datetime, $duration_minutes = 60 ) {
		// 1. Check teacher's general working hours (Mocked for now)
		// In a real scenario, this would fetch from user meta or a separate table.
		$day_of_week = date( 'l', strtotime( $datetime ) );
		$hour = date( 'H', strtotime( $datetime ) );

		// Example: Available Mon-Fri, 9am - 5pm
		if ( in_array( $day_of_week, array( 'Saturday', 'Sunday' ) ) ) {
			return false;
		}
		if ( $hour < 9 || $hour >= 17 ) {
			return false;
		}

		// 2. Check for conflicts with existing sessions (Group or Private)
		if ( $this->has_conflict( $teacher_id, $datetime, $duration_minutes ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Check for scheduling conflicts.
	 *
	 * @param int $teacher_id
	 * @param string $start_time
	 * @param int $duration_minutes
	 * @return bool True if conflict exists
	 */
	public function has_conflict( $teacher_id, $start_time, $duration_minutes ) {
		global $wpdb;
		$table_sessions = $wpdb->prefix . 'mara7el_sessions';

		$start = date( 'Y-m-d H:i:s', strtotime( $start_time ) );
		$end = date( 'Y-m-d H:i:s', strtotime( $start_time ) + ( $duration_minutes * 60 ) );

		// Check against sessions table
		// We need to join with groups/private classes to filter by teacher_id, 
		// but for simplicity, let's assume we query sessions directly if we store teacher_id there?
		// The schema for sessions has reference_id. We need to look up the teacher from the reference.
		
		// Optimization: It's better to query sessions and join.
		// However, for this MVP, let's do a slightly more complex query.
		
		$query = "
			SELECT count(*) FROM $table_sessions s
			LEFT JOIN {$wpdb->prefix}mara7el_groups g ON (s.type = 'group' AND s.reference_id = g.id)
			LEFT JOIN {$wpdb->prefix}mara7el_private_classes p ON (s.type = 'private' AND s.reference_id = p.id)
			WHERE 
				(g.teacher_id = %d OR p.teacher_id = %d)
				AND s.session_date < %s 
				AND DATE_ADD(s.session_date, INTERVAL 60 MINUTE) > %s
				AND s.status != 'cancelled'
		";
		
		// Note: Assuming all sessions are 60 mins for conflict check simplicity, 
		// or we should store duration in sessions table. 
		// For MVP, let's assume 60 mins.

		$count = $wpdb->get_var( $wpdb->prepare( $query, $teacher_id, $teacher_id, $end, $start ) );

		return $count > 0;
	}

	/**
	 * Generate available slots for a teacher and course.
	 *
	 * @param int $teacher_id
	 * @param int $course_id
	 * @param string $start_date Y-m-d
	 * @param string $end_date Y-m-d
	 * @return array
	 */
	public function get_available_slots( $teacher_id, $course_id, $start_date, $end_date ) {
		$slots = array();
		$current = strtotime( $start_date );
		$end = strtotime( $end_date );

		while ( $current <= $end ) {
			// Check 9am to 5pm slots
			for ( $h = 9; $h < 17; $h++ ) {
				$time = date( 'Y-m-d', $current ) . " $h:00:00";
				if ( $this->is_teacher_available( $teacher_id, $time ) ) {
					$slots[] = $time;
				}
			}
			$current = strtotime( '+1 day', $current );
		}

		return $slots;
	}
}
