<?php
/**
 * Admin Settings Class.
 *
 * @package Mara7el_Live
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Mara7el_Live_Admin_Settings {

	/**
	 * Constructor.
	 */
	public function __construct() {
		// Use later priority to ensure LearnPress menu exists
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ), 100 );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
	}
	
	/**
	 * Enqueue admin scripts for settings page.
	 */
	public function enqueue_admin_scripts( $hook ) {
		// Only load on our settings page
		if ( 'learnpress_page_mara7el-live-settings' !== $hook ) {
			return;
		}
		
		// Add inline script for tab switching
		$script = "
		jQuery(document).ready(function($) {
			$('.nav-tab').click(function(e) {
				e.preventDefault();
				$('.nav-tab').removeClass('nav-tab-active');
				$(this).addClass('nav-tab-active');
				$('.tab-content').hide();
				var target = $(this).attr('href') + '-tab';
				$(target).show();
			});
		});
		";
		wp_add_inline_script( 'jquery', $script );
	}

	/**
	 * Add admin menu page.
	 */
	public function add_admin_menu() {
		// Verify LearnPress menu exists
		global $menu;
		$lp_menu_exists = false;
		
		if ( is_array( $menu ) ) {
			foreach ( $menu as $item ) {
				if ( isset( $item[2] ) && $item[2] === 'learn_press' ) {
					$lp_menu_exists = true;
					break;
				}
			}
		}
		
		// Add as submenu under LearnPress
		if ( $lp_menu_exists ) {
			add_submenu_page(
				'learn_press',
				__( 'Live Classes Settings', 'mara7el-live' ),
				__( 'Live Classes', 'mara7el-live' ),
				'manage_options',
				'mara7el-live-settings',
				array( $this, 'settings_page_html' ),
				30
			);
		} else {
			// Fallback: Add as top-level menu if LearnPress menu doesn't exist
			add_menu_page(
				__( 'Live Classes Settings', 'mara7el-live' ),
				__( 'Live Classes', 'mara7el-live' ),
				'manage_options',
				'mara7el-live-settings',
				array( $this, 'settings_page_html' ),
				'dashicons-video-alt2',
				58
			);
		}
	}

	/**
	 * Register settings.
	 */
	public function register_settings() {
		// Zoom Settings
		register_setting( 'mara7el_live_options_group', 'mara7el_zoom_api_key' );
		register_setting( 'mara7el_live_options_group', 'mara7el_zoom_api_secret' );
		
		// Google Meet Settings
		register_setting( 'mara7el_live_options_group', 'mara7el_google_client_id' );
		register_setting( 'mara7el_live_options_group', 'mara7el_google_client_secret' );

		// Microsoft Teams Settings
		register_setting( 'mara7el_live_options_group', 'mara7el_teams_client_id' );
		register_setting( 'mara7el_live_options_group', 'mara7el_teams_client_secret' );
		register_setting( 'mara7el_live_options_group', 'mara7el_teams_tenant_id' );
	}

	/**
	 * Settings page HTML.
	 */
	public function settings_page_html() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		?>
		<div class="wrap">
			<h1><?php echo esc_html__( 'Live Classes for LearnPress - Settings', 'mara7el-live' ); ?></h1>
			<p class="description"><?php _e( 'Configure your live session platform credentials to integrate Zoom, Google Meet, or Microsoft Teams with LearnPress courses and lessons.', 'mara7el-live' ); ?></p>
			<form action="options.php" method="post">
				<?php
				settings_fields( 'mara7el_live_options_group' );
				?>
				<h2 class="nav-tab-wrapper">
					<a href="#zoom" class="nav-tab nav-tab-active">Zoom</a>
					<a href="#google" class="nav-tab">Google Meet</a>
					<a href="#teams" class="nav-tab">Microsoft Teams</a>
				</h2>

				<div id="zoom-tab" class="tab-content">
					<h3>Zoom API Settings</h3>
					<table class="form-table">
						<tr valign="top">
							<th scope="row">API Key (Client ID)</th>
							<td><input type="text" name="mara7el_zoom_api_key" value="<?php echo esc_attr( get_option( 'mara7el_zoom_api_key' ) ); ?>" class="regular-text" /></td>
						</tr>
						<tr valign="top">
							<th scope="row">API Secret (Client Secret)</th>
							<td><input type="password" name="mara7el_zoom_api_secret" value="<?php echo esc_attr( get_option( 'mara7el_zoom_api_secret' ) ); ?>" class="regular-text" /></td>
						</tr>
					</table>
					<p class="description">Enter your Zoom Server-to-Server OAuth credentials.</p>
				</div>

				<div id="google-tab" class="tab-content" style="display:none;">
					<h3>Google Meet API Settings</h3>
					<table class="form-table">
						<tr valign="top">
							<th scope="row">Client ID</th>
							<td><input type="text" name="mara7el_google_client_id" value="<?php echo esc_attr( get_option( 'mara7el_google_client_id' ) ); ?>" class="regular-text" /></td>
						</tr>
						<tr valign="top">
							<th scope="row">Client Secret</th>
							<td><input type="password" name="mara7el_google_client_secret" value="<?php echo esc_attr( get_option( 'mara7el_google_client_secret' ) ); ?>" class="regular-text" /></td>
						</tr>
					</table>
				</div>

				<div id="teams-tab" class="tab-content" style="display:none;">
					<h3>Microsoft Teams API Settings</h3>
					<table class="form-table">
						<tr valign="top">
							<th scope="row">Client ID</th>
							<td><input type="text" name="mara7el_teams_client_id" value="<?php echo esc_attr( get_option( 'mara7el_teams_client_id' ) ); ?>" class="regular-text" /></td>
						</tr>
						<tr valign="top">
							<th scope="row">Client Secret</th>
							<td><input type="password" name="mara7el_teams_client_secret" value="<?php echo esc_attr( get_option( 'mara7el_teams_client_secret' ) ); ?>" class="regular-text" /></td>
						</tr>
						<tr valign="top">
							<th scope="row">Tenant ID</th>
							<td><input type="text" name="mara7el_teams_tenant_id" value="<?php echo esc_attr( get_option( 'mara7el_teams_tenant_id' ) ); ?>" class="regular-text" /></td>
						</tr>
					</table>
				</div>

				<?php submit_button( 'Save Settings' ); ?>
			</form>
		</div>
		<?php
	}
}
