<?php

/**
 * LearnPress Hooks Class
 * 
 * Handles all frontend LearnPress integration including:
 * - Course page booking buttons
 * - Enrollment checking
 * - User role validation
 */
class Mara7el_Booking_LearnPress_Hooks {

	/**
	 * Integration instance
	 */
	private $lp_integration;

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->lp_integration = Mara7el_Booking_LearnPress_Integration::get_instance();
	}

	/**
	 * Initialize hooks.
	 */
	public function init() {
		// Only proceed if LearnPress is active
		if ( ! $this->lp_integration->is_learnpress_active() ) {
			return;
		}

		// Add booking buttons to LearnPress course pages
		add_action( 'learn-press/after-single-course-summary', array( $this, 'render_booking_buttons' ), 20 );
		
		// Alternative hook for different themes
		add_action( 'learn-press/single-course-summary', array( $this, 'render_booking_buttons' ), 90 );

		// Enqueue frontend assets
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );
		
		// Add shortcodes for teacher and student dashboards
		add_shortcode( 'mara7el_teacher_dashboard', array( $this, 'teacher_dashboard_shortcode' ) );
		add_shortcode( 'mara7el_student_dashboard', array( $this, 'student_dashboard_shortcode' ) );
	}

	/**
	 * Enqueue frontend assets.
	 */
	public function enqueue_assets() {
		// Only load on LearnPress course pages or pages with shortcodes
		if ( ! is_singular( 'lp_course' ) && ! $this->has_booking_shortcode() ) {
			return;
		}

		wp_enqueue_style( 
			'mara7el-booking-css', 
			MARA7EL_BOOKING_URL . 'assets/css/booking.css', 
			array(), 
			MARA7EL_BOOKING_VERSION 
		);
		
		wp_enqueue_script( 
			'mara7el-booking-js', 
			MARA7EL_BOOKING_URL . 'assets/js/booking.js', 
			array( 'jquery' ), 
			MARA7EL_BOOKING_VERSION, 
			true 
		);

		wp_localize_script( 'mara7el-booking-js', 'mara7el_booking_vars', array(
			'api_url' => esc_url_raw( rest_url( 'mara7el-booking/v1/' ) ),
			'nonce'   => wp_create_nonce( 'wp_rest' ),
			'strings' => array(
				'login_required' => __( 'Please login to book a class.', 'mara7el-booking' ),
				'enroll_required' => __( 'Please enroll in this course first.', 'mara7el-booking' ),
				'booking_success' => __( 'Booking successful!', 'mara7el-booking' ),
				'booking_error'   => __( 'Booking failed. Please try again.', 'mara7el-booking' ),
			),
		) );
	}

	/**
	 * Check if current post has booking shortcodes
	 */
	private function has_booking_shortcode() {
		global $post;
		
		if ( ! $post ) {
			return false;
		}

		return has_shortcode( $post->post_content, 'mara7el_teacher_dashboard' ) 
			|| has_shortcode( $post->post_content, 'mara7el_student_dashboard' );
	}

	/**
	 * Render booking buttons on course pages.
	 */
	public function render_booking_buttons() {
		// Only show on single course pages
		if ( ! is_singular( 'lp_course' ) ) {
			return;
		}

		$course_id = get_the_ID();
		
		// Check if user is logged in
		if ( ! is_user_logged_in() ) {
			echo '<div class="mara7el-booking-notice">';
			echo '<p>' . esc_html__( 'Please login to book a class.', 'mara7el-booking' ) . '</p>';
			echo '</div>';
			return;
		}

		$user_id = get_current_user_id();

		// Check if user is enrolled in the course
		if ( ! $this->lp_integration->is_user_enrolled( $user_id, $course_id ) ) {
			echo '<div class="mara7el-booking-notice">';
			echo '<p>' . esc_html__( 'Please enroll in this course to book a class.', 'mara7el-booking' ) . '</p>';
			echo '</div>';
			return;
		}

		// Check if user is an instructor (instructors shouldn't book their own courses)
		if ( $this->lp_integration->is_instructor( $user_id ) ) {
			$instructors = $this->lp_integration->get_course_instructors( $course_id );
			if ( in_array( $user_id, $instructors ) ) {
				return; // Don't show booking for course instructor
			}
		}

		// Load the booking buttons template
		$this->load_booking_template( $course_id );
	}

	/**
	 * Load booking template
	 */
	private function load_booking_template( $course_id ) {
		$template_path = MARA7EL_BOOKING_PATH . 'templates/booking-modal.php';
		
		if ( file_exists( $template_path ) ) {
			include $template_path;
		} else {
			// Fallback inline template
			?>
			<div class="mara7el-booking-buttons">
				<h3><?php esc_html_e( 'Book a Class', 'mara7el-booking' ); ?></h3>
				<button class="mara7el-booking-btn mara7el-booking-group" data-course-id="<?php echo esc_attr( $course_id ); ?>">
					<?php esc_html_e( 'Join Group Class (Max 3 Students)', 'mara7el-booking' ); ?>
				</button>
				<button class="mara7el-booking-btn mara7el-booking-private" data-course-id="<?php echo esc_attr( $course_id ); ?>">
					<?php esc_html_e( 'Book Private One-to-One Class', 'mara7el-booking' ); ?>
				</button>
			</div>
			<?php
		}
	}

	/**
	 * Teacher dashboard shortcode
	 */
	public function teacher_dashboard_shortcode( $atts ) {
		if ( ! is_user_logged_in() ) {
			return '<p>' . esc_html__( 'Please login to view your dashboard.', 'mara7el-booking' ) . '</p>';
		}

		if ( ! $this->lp_integration->is_instructor( get_current_user_id() ) ) {
			return '<p>' . esc_html__( 'This dashboard is only available for instructors.', 'mara7el-booking' ) . '</p>';
		}

		ob_start();
		$teacher_ui = new Mara7el_Booking_Teacher_UI();
		echo $teacher_ui->render_dashboard();
		return ob_get_clean();
	}

	/**
	 * Student dashboard shortcode
	 */
	public function student_dashboard_shortcode( $atts ) {
		if ( ! is_user_logged_in() ) {
			return '<p>' . esc_html__( 'Please login to view your bookings.', 'mara7el-booking' ) . '</p>';
		}

		ob_start();
		$student_ui = new Mara7el_Booking_Student_UI();
		echo $student_ui->render_dashboard();
		return ob_get_clean();
	}
}
