<?php

/**
 * LearnPress Integration Class
 * 
 * Handles all LearnPress-specific integration including:
 * - Dependency checking
 * - Addon registration
 * - Version validation
 * - Admin menu integration
 */
class Mara7el_Booking_LearnPress_Integration {

	/**
	 * Minimum required LearnPress version
	 */
	const MIN_LP_VERSION = '4.3.1';

	/**
	 * Instance of this class
	 */
	private static $instance = null;

	/**
	 * Get singleton instance
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		add_action( 'plugins_loaded', array( $this, 'check_learnpress_dependency' ), 1 );
		add_action( 'admin_notices', array( $this, 'display_admin_notices' ) );
		
		// Register as LearnPress addon
		add_filter( 'learn-press/addon-list', array( $this, 'register_addon' ) );
	}

	/**
	 * Check if LearnPress is installed and activated
	 */
	public function check_learnpress_dependency() {
		if ( ! $this->is_learnpress_active() ) {
			add_action( 'admin_notices', array( $this, 'learnpress_missing_notice' ) );
			return false;
		}

		if ( ! $this->is_learnpress_version_compatible() ) {
			add_action( 'admin_notices', array( $this, 'learnpress_version_notice' ) );
			return false;
		}

		return true;
	}

	/**
	 * Check if LearnPress is active
	 */
	public function is_learnpress_active() {
		return class_exists( 'LearnPress' );
	}

	/**
	 * Check if LearnPress version is compatible
	 */
	public function is_learnpress_version_compatible() {
		if ( ! $this->is_learnpress_active() ) {
			return false;
		}

		$lp_version = defined( 'LEARNPRESS_VERSION' ) ? LEARNPRESS_VERSION : '0.0.0';
		return version_compare( $lp_version, self::MIN_LP_VERSION, '>=' );
	}

	/**
	 * Display admin notice if LearnPress is not active
	 */
	public function learnpress_missing_notice() {
		?>
		<div class="notice notice-error">
			<p>
				<strong><?php esc_html_e( 'Mara7el Booking - LearnPress Addon', 'mara7el-booking' ); ?></strong>
				<?php esc_html_e( 'requires LearnPress to be installed and activated.', 'mara7el-booking' ); ?>
				<a href="<?php echo esc_url( admin_url( 'plugin-install.php?s=learnpress&tab=search&type=term' ) ); ?>">
					<?php esc_html_e( 'Install LearnPress now', 'mara7el-booking' ); ?>
				</a>
			</p>
		</div>
		<?php
	}

	/**
	 * Display admin notice if LearnPress version is incompatible
	 */
	public function learnpress_version_notice() {
		$lp_version = defined( 'LEARNPRESS_VERSION' ) ? LEARNPRESS_VERSION : '0.0.0';
		?>
		<div class="notice notice-warning">
			<p>
				<strong><?php esc_html_e( 'Mara7el Booking - LearnPress Addon', 'mara7el-booking' ); ?></strong>
				<?php
				printf(
					/* translators: 1: Required version, 2: Current version */
					esc_html__( 'requires LearnPress version %1$s or higher. You are currently using version %2$s. Please update LearnPress.', 'mara7el-booking' ),
					esc_html( self::MIN_LP_VERSION ),
					esc_html( $lp_version )
				);
				?>
			</p>
		</div>
		<?php
	}

	/**
	 * Display any admin notices
	 */
	public function display_admin_notices() {
		// Additional notices can be added here
	}

	/**
	 * Register this plugin as a LearnPress addon
	 */
	public function register_addon( $addons ) {
		$addons['mara7el-booking'] = array(
			'name'        => __( 'Mara7el Booking', 'mara7el-booking' ),
			'description' => __( 'Complete Class Booking System for LearnPress - Group & Private Classes', 'mara7el-booking' ),
			'version'     => MARA7EL_BOOKING_VERSION,
			'author'      => 'Mara7el Team',
			'url'         => 'https://www.mara7el.com/',
			'icon'        => MARA7EL_BOOKING_URL . 'assets/images/icon.png',
		);
		return $addons;
	}

	/**
	 * Get LearnPress course by ID
	 */
	public function get_course( $course_id ) {
		if ( ! $this->is_learnpress_active() ) {
			return null;
		}
		return learn_press_get_course( $course_id );
	}

	/**
	 * Get LearnPress user
	 */
	public function get_user( $user_id = null ) {
		if ( ! $this->is_learnpress_active() ) {
			return null;
		}
		return learn_press_get_user( $user_id );
	}

	/**
	 * Check if user is a LearnPress instructor
	 */
	public function is_instructor( $user_id = null ) {
		if ( ! $this->is_learnpress_active() ) {
			return false;
		}
		
		$user_id = $user_id ? $user_id : get_current_user_id();
		$user = get_userdata( $user_id );
		
		if ( ! $user ) {
			return false;
		}

		return in_array( 'lp_teacher', $user->roles ) || user_can( $user_id, 'manage_options' );
	}

	/**
	 * Check if user is enrolled in a course
	 */
	public function is_user_enrolled( $user_id, $course_id ) {
		if ( ! $this->is_learnpress_active() ) {
			return false;
		}

		$user = $this->get_user( $user_id );
		if ( ! $user ) {
			return false;
		}

		return $user->has_enrolled_course( $course_id );
	}

	/**
	 * Get course instructors
	 */
	public function get_course_instructors( $course_id ) {
		if ( ! $this->is_learnpress_active() ) {
			return array();
		}

		$course = $this->get_course( $course_id );
		if ( ! $course ) {
			return array();
		}

		// Get course author (main instructor)
		$author_id = $course->get_author()->ID;
		
		return array( $author_id );
	}

	/**
	 * Get all instructors
	 */
	public function get_all_instructors() {
		$args = array(
			'role'    => 'lp_teacher',
			'orderby' => 'display_name',
		);
		
		$instructors = get_users( $args );
		
		// Also include admins
		$admins = get_users( array( 'role' => 'administrator' ) );
		
		return array_merge( $instructors, $admins );
	}

	/**
	 * Get course title
	 */
	public function get_course_title( $course_id ) {
		$course = $this->get_course( $course_id );
		return $course ? $course->get_title() : sprintf( __( 'Course #%d', 'mara7el-booking' ), $course_id );
	}

	/**
	 * Get user display name
	 */
	public function get_user_display_name( $user_id ) {
		$user = get_userdata( $user_id );
		return $user ? $user->display_name : sprintf( __( 'User #%d', 'mara7el-booking' ), $user_id );
	}
}

// Initialize the integration
Mara7el_Booking_LearnPress_Integration::get_instance();
