<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Mara7el_Watermark_Token_Manager
 * Handles generation and validation of access tokens.
 */
class Mara7el_Watermark_Token_Manager {

	/**
	 * Secret key for signing.
	 *
	 * @var string
	 */
	private $secret_key;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->secret_key = defined( 'AUTH_KEY' ) ? AUTH_KEY : 'mara7el-watermark-secret';
	}

	/**
	 * Generate a signed token.
	 *
	 * @param int    $user_id  User ID.
	 * @param string $file_id  File ID/Path.
	 * @param int    $ttl      Time to live in seconds.
	 * @return string The token.
	 */
	public function generate_token( $user_id, $file_id, $ttl = 3600 ) {
		$expiry = time() + $ttl;
		$data = array(
			'u' => $user_id,
			'f' => $file_id,
			'e' => $expiry,
			's' => wp_create_nonce( 'mara7el_download_' . $user_id . '_' . $file_id ),
		);

		$payload = base64_encode( json_encode( $data ) );
		$signature = hash_hmac( 'sha256', $payload, $this->secret_key );

		return $payload . '.' . $signature;
	}

	/**
	 * Validate a token.
	 *
	 * @param string $token The token string.
	 * @return array|WP_Error Decoded data or error.
	 */
	public function validate_token( $token ) {
		$parts = explode( '.', $token );
		if ( count( $parts ) !== 2 ) {
			return new WP_Error( 'invalid_token', 'Invalid token format.' );
		}

		list( $payload, $signature ) = $parts;

		// Verify signature
		$expected_signature = hash_hmac( 'sha256', $payload, $this->secret_key );
		if ( ! hash_equals( $expected_signature, $signature ) ) {
			return new WP_Error( 'invalid_signature', 'Token signature mismatch.' );
		}

		$data = json_decode( base64_decode( $payload ), true );
		if ( ! $data ) {
			return new WP_Error( 'invalid_payload', 'Could not decode token payload.' );
		}

		// Check expiry
		if ( isset( $data['e'] ) && time() > $data['e'] ) {
			return new WP_Error( 'token_expired', 'Token has expired.' );
		}

		// Check nonce (optional, binds to session)
		if ( isset( $data['s'] ) && ! wp_verify_nonce( $data['s'], 'mara7el_download_' . $data['u'] . '_' . $data['f'] ) ) {
			// Nonce check might fail if session changed or logged out, 
			// but for download links we might want to be lenient or strict depending on requirements.
			// For now, we'll log it but maybe allow if strict mode is off?
			// Let's enforce it for security.
			return new WP_Error( 'invalid_nonce', 'Invalid nonce.' );
		}

		return $data;
	}
}
