<?php

use setasign\Fpdi\Fpdi;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Mara7el_Watermark_Engine
 * Handles the application of watermarks to files.
 */
class Mara7el_Watermark_Engine {

	/**
	 * Apply watermark to a file.
	 *
	 * @param string $file_path Absolute path to the source file.
	 * @param int    $user_id   User ID to watermark for.
	 * @return string|WP_Error Path to the watermarked file (cached) or error.
	 */
	public function apply_watermark( $file_path, $user_id ) {
		if ( ! file_exists( $file_path ) ) {
			return new WP_Error( 'file_not_found', 'File not found.' );
		}

		$extension = strtolower( pathinfo( $file_path, PATHINFO_EXTENSION ) );
		$settings = get_option( 'mara7el_watermark_settings', array() );
		
		// Check if watermarking is enabled
		if ( empty( $settings['enable_watermark'] ) ) {
			return $file_path; // Return original if disabled
		}

		// Check cache first
		$cache_manager = new Mara7el_Watermark_Cache_Manager();
		$cache_key = $cache_manager->generate_cache_key( $file_path, $user_id, $settings );
		$cached_file = $cache_manager->get_cached_file( $cache_key, $extension );

		if ( $cached_file ) {
			return $cached_file;
		}

		// Generate watermark text
		$text = $this->get_watermark_text( $user_id, $settings );

		// Process based on type
		$result = false;
		if ( in_array( $extension, array( 'jpg', 'jpeg', 'png' ) ) ) {
			$result = $this->watermark_image( $file_path, $text, $settings );
		} elseif ( $extension === 'pdf' ) {
			$result = $this->watermark_pdf( $file_path, $text, $settings );
		} else {
			// Unsupported type for watermarking, return original
			return $file_path;
		}

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		// Store in cache
		if ( $result ) {
			$stored_path = $cache_manager->store_cached_file( $cache_key, $result, $extension );
			return $stored_path ? $stored_path : new WP_Error( 'cache_error', 'Failed to store cached file.' );
		}

		return new WP_Error( 'processing_error', 'Failed to process watermark.' );
	}

	/**
	 * Generate watermark text with placeholders replaced.
	 */
	private function get_watermark_text( $user_id, $settings ) {
		$user = get_userdata( $user_id );
		$template = isset( $settings['watermark_text'] ) ? $settings['watermark_text'] : 'User: {user_id}';

		$replacements = array(
			'{user_id}'    => $user_id,
			'{user_name}'  => $user ? $user->display_name : 'Guest',
			'{user_email}' => $user ? $user->user_email : 'unknown',
			'{date}'       => date_i18n( get_option( 'date_format' ) ),
			'{time}'       => date_i18n( get_option( 'time_format' ) ),
			'{ip_address}' => isset( $_SERVER['REMOTE_ADDR'] ) ? $_SERVER['REMOTE_ADDR'] : '',
		);

		return str_replace( array_keys( $replacements ), array_values( $replacements ), $template );
	}

	/**
	 * Watermark an image using GD.
	 */
	private function watermark_image( $file_path, $text, $settings ) {
		$info = getimagesize( $file_path );
		$mime = $info['mime'];

		switch ( $mime ) {
			case 'image/jpeg':
				$image = imagecreatefromjpeg( $file_path );
				break;
			case 'image/png':
				$image = imagecreatefrompng( $file_path );
				break;
			default:
				return new WP_Error( 'invalid_image', 'Unsupported image type.' );
		}

		$color_val = 255; // White
		// Allocate color (white)
		$color = imagecolorallocatealpha( $image, 255, 255, 255, ( 100 - (int)$settings['opacity'] ) ); // 0-127 alpha in GD
		
		// Add text (using built-in font for simplicity, could use TTF if font file provided)
		// GD built-in fonts are small, might need imagettftext if available
		if ( function_exists( 'imagettftext' ) ) {
			// Use a standard font if available, or fallback to built-in
			// For this example, we'll try to find a font or fail gracefully to built-in
			$font_size = isset( $settings['font_size'] ) ? (int)$settings['font_size'] : 12;
			// Path to a font file - using a system font might be tricky, let's use a bundled one if we had it.
			// For now, use built-in string function as fallback
			imagestring( $image, 5, 10, 10, $text, $color );
		} else {
			imagestring( $image, 5, 10, 10, $text, $color );
		}

		ob_start();
		if ( $mime === 'image/jpeg' ) {
			imagejpeg( $image );
		} else {
			imagepng( $image );
		}
		$content = ob_get_clean();
		imagedestroy( $image );

		return $content;
	}

	/**
	 * Watermark a PDF using FPDI.
	 */
	private function watermark_pdf( $file_path, $text, $settings ) {
		if ( ! class_exists( '\setasign\Fpdi\Fpdi' ) ) {
			// Fallback or error if library missing
			return new WP_Error( 'missing_dependency', 'FPDI library not found. Please run composer install.' );
		}

		try {
			$pdf = new Fpdi();
			$page_count = $pdf->setSourceFile( $file_path );

			for ( $page_no = 1; $page_no <= $page_count; $page_no++ ) {
				$template_id = $pdf->importPage( $page_no );
				$size = $pdf->getTemplateSize( $template_id );

				$pdf->AddPage( $size['orientation'], array( $size['width'], $size['height'] ) );
				$pdf->useTemplate( $template_id );

				$pdf->SetFont( 'Helvetica', '', (int)$settings['font_size'] );
				$pdf->SetTextColor( 200, 200, 200 ); // Light gray
				$pdf->SetXY( 10, 10 ); // Top left
				$pdf->Write( 0, $text );
				
				// Diagonal watermark
				$pdf->SetXY( $size['width'] / 2, $size['height'] / 2 );
				// Rotation is harder in pure FPDF without extensions, but we can just place text
				// FPDI doesn't support rotation natively on imported pages easily without extensions
			}

			return $pdf->Output( 'S' ); // Return as string
		} catch ( Exception $e ) {
			return new WP_Error( 'pdf_error', $e->getMessage() );
		}
	}
}
