<?php
/**
 * Plugin Name: Mara7el Live Classes for LearnPress
 * Plugin URI:  https://example.com/mara7el-live-classes
 * Description: LearnPress addon that integrates courses and lessons with external live-session platforms (Zoom, Google Meet, Microsoft Teams).
 * Version:     2.0.0
 * Author:      Mara7el
 * Author URI:  https://example.com
 * Text Domain: mara7el-live
 * Domain Path: /languages
 * Requires Plugins: learnpress
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * LearnPress Requires: 4.0.0
 * LearnPress Tested up to: 4.3.1
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

// Define plugin constants.
define( 'MARA7EL_LIVE_VERSION', '2.0.0' );
define( 'MARA7EL_LIVE_PATH', plugin_dir_path( __FILE__ ) );
define( 'MARA7EL_LIVE_URL', plugin_dir_url( __FILE__ ) );
define( 'MARA7EL_LIVE_LEARNPRESS_MIN_VERSION', '4.0.0' );

/**
 * Check if LearnPress is active and compatible.
 *
 * Uses multiple detection methods for maximum compatibility with different
 * WordPress/hosting environments.
 *
 * @since 2.0.0
 * @return bool
 */
function mara7el_live_check_learnpress() {
	// Method 1: Check if LearnPress main class exists
	if ( class_exists( 'LearnPress' ) || class_exists( 'LP_Core' ) ) {
		return mara7el_live_check_learnpress_version();
	}

	// Method 2: Check if LearnPress function exists
	if ( function_exists( 'learn_press_get_version' ) ) {
		return mara7el_live_check_learnpress_version();
	}

	// Method 3: Check if LearnPress constant is defined
	if ( defined( 'LP_PLUGIN_FILE' ) || defined( 'LEARNPRESS_VERSION' ) ) {
		return mara7el_live_check_learnpress_version();
	}

	// Method 4: Check if LearnPress plugin file is active (fallback for early loading)
	if ( ! function_exists( 'is_plugin_active' ) ) {
		include_once ABSPATH . 'wp-admin/includes/plugin.php';
	}

	$learnpress_plugins = array(
		'learnpress/learnpress.php',
		'LearnPress/learnpress.php',
		'learn-press/learnpress.php',
	);

	foreach ( $learnpress_plugins as $plugin ) {
		if ( is_plugin_active( $plugin ) ) {
			return mara7el_live_check_learnpress_version();
		}
	}

	return false;
}

/**
 * Check LearnPress version compatibility.
 *
 * @since 2.0.0
 * @return bool
 */
function mara7el_live_check_learnpress_version() {
	// Try to get version from function
	if ( function_exists( 'learn_press_get_version' ) ) {
		$lp_version = learn_press_get_version();
		if ( version_compare( $lp_version, MARA7EL_LIVE_LEARNPRESS_MIN_VERSION, '<' ) ) {
			return false;
		}
		return true;
	}

	// Try to get version from constant
	if ( defined( 'LEARNPRESS_VERSION' ) ) {
		if ( version_compare( LEARNPRESS_VERSION, MARA7EL_LIVE_LEARNPRESS_MIN_VERSION, '<' ) ) {
			return false;
		}
		return true;
	}

	// Try to get version from LearnPress class
	if ( class_exists( 'LearnPress' ) && method_exists( 'LearnPress', 'instance' ) ) {
		$lp = LearnPress::instance();
		if ( isset( $lp->version ) ) {
			if ( version_compare( $lp->version, MARA7EL_LIVE_LEARNPRESS_MIN_VERSION, '<' ) ) {
				return false;
			}
			return true;
		}
	}

	// If we can't determine version but LearnPress is detected, assume compatible
	return true;
}

/**
 * Check if LearnPress is detected (any method).
 *
 * @since 2.0.0
 * @return bool
 */
function mara7el_live_is_learnpress_detected() {
	if ( class_exists( 'LearnPress' ) || class_exists( 'LP_Core' ) ) {
		return true;
	}
	if ( function_exists( 'learn_press_get_version' ) ) {
		return true;
	}
	if ( defined( 'LP_PLUGIN_FILE' ) || defined( 'LEARNPRESS_VERSION' ) ) {
		return true;
	}
	return false;
}

/**
 * Display admin notice if LearnPress is not active or incompatible.
 *
 * @since 2.0.0
 */
function mara7el_live_learnpress_missing_notice() {
	// Double-check - maybe LearnPress loaded later
	if ( mara7el_live_check_learnpress() ) {
		return; // LearnPress is now detected, don't show notice
	}

	$class = 'notice notice-error';
	
	if ( ! mara7el_live_is_learnpress_detected() ) {
		$message = sprintf(
			/* translators: %s: LearnPress plugin link */
			__( 'Mara7el Live Classes requires LearnPress to be installed and activated. Please install %s first.', 'mara7el-live' ),
			'<a href="https://wordpress.org/plugins/learnpress/" target="_blank">LearnPress</a>'
		);
	} else {
		// LearnPress detected but version incompatible
		$lp_version = 'unknown';
		if ( function_exists( 'learn_press_get_version' ) ) {
			$lp_version = learn_press_get_version();
		} elseif ( defined( 'LEARNPRESS_VERSION' ) ) {
			$lp_version = LEARNPRESS_VERSION;
		}
		$message = sprintf(
			/* translators: %1$s: current LearnPress version, %2$s: required LearnPress version */
			__( 'Mara7el Live Classes requires LearnPress version %2$s or higher. You are using LearnPress %1$s. Please update LearnPress.', 'mara7el-live' ),
			$lp_version,
			MARA7EL_LIVE_LEARNPRESS_MIN_VERSION
		);
	}

	printf( '<div class="%1$s"><p>%2$s</p></div>', esc_attr( $class ), wp_kses_post( $message ) );
}

/**
 * Initialize the plugin.
 * 
 * Runs on 'plugins_loaded' with priority 20 to ensure LearnPress loads first.
 *
 * @since 2.0.0
 */
function mara7el_live_init() {
	// Check LearnPress activation before loading plugin.
	if ( ! mara7el_live_check_learnpress() ) {
		add_action( 'admin_notices', 'mara7el_live_learnpress_missing_notice' );
		return; // Stop plugin initialization.
	}

	// Include the main class loader.
	require_once MARA7EL_LIVE_PATH . 'includes/class-loader.php';

	// Initialize the plugin immediately - the loader will handle LearnPress integration
	$GLOBALS['mara7el_live'] = mara7el_live();
}

/**
 * Main instance of Mara7el Live Classes.
 *
 * Returns the main instance of Mara7el Live Classes to prevent the need to use globals.
 *
 * @since  1.0.0
 * @return Mara7el_Live_Loader
 */
function mara7el_live() {
	return Mara7el_Live_Loader::instance();
}

/**
 * Late initialization fallback.
 * 
 * Some hosting environments may load plugins in unexpected order.
 * This provides a fallback initialization on 'init' hook.
 *
 * @since 2.0.0
 */
function mara7el_live_late_init() {
	// Only run if not already initialized
	if ( isset( $GLOBALS['mara7el_live'] ) ) {
		return;
	}

	// Try to initialize if LearnPress is now available
	if ( mara7el_live_check_learnpress() ) {
		require_once MARA7EL_LIVE_PATH . 'includes/class-loader.php';
		$GLOBALS['mara7el_live'] = mara7el_live();
		
		// Remove the admin notice since we successfully initialized
		remove_action( 'admin_notices', 'mara7el_live_learnpress_missing_notice' );
	}
}

// Hook into plugins_loaded with priority 30 to ensure LearnPress loads first
// (LearnPress typically uses priority 10-20)
add_action( 'plugins_loaded', 'mara7el_live_init', 30 );

// Fallback: Try again on 'init' hook in case LearnPress loaded late
add_action( 'init', 'mara7el_live_late_init', 0 );
